%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                   %
% I/O operations for SModels for Prolog                             %
% Thomas Linke, Christian Anger, June 2000                          %        
%                                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%
% read_LP( +File, -LP ) true iff LP is Logic Program read from file File
%
read_LP( File, LP ):- 
	exists(File) -> %if
	( % then read file
	    open(File,read,Stream),
	    read_clauses(Stream,LP),
	    close(Stream),!
        );
	( % else try 'file.kb'
	    atom_chars(File,FileChars),
	    append(FileChars,".kb",Filename),
	    atom_chars(NewFile,Filename),
	    open(NewFile,read,Stream),
	    read_clauses(Stream,LP),
	    close(Stream),!
        ).


read_clauses( Stream, [C|Rest] ):-
	read(Stream,C),
	not(C = end_of_file),
	read_clauses(Stream,Rest).
read_clauses( _, [] ):-!.

%
% translation between external  and internal format
% ext2int(E,I) true iff E is external and I is internal representation of a 
% logic program
% internal representation of clause    p:- q,s,not r,not f.
%                    is prolog term    rule(p,[q,s],[r,f])
% rule(Cons,Pos,Neg) represents a clause with positive body-literals Pos,
% negative body literals Neg (Pos and Neg are lists) and head Cons.
%
ext2int([],[]):-!.
ext2int([C|E],[L|I]):-
	list_representation(C,L),!, % need only one solution
	ext2int(E,I).

%
% list_representation(+P,-L) true iff
% P is prolog Term
% L is list-respresentation of P
%
list_representation( (Head :- Body) , rule(Head,Pos,Neg) ):-
	split_body(Body,Pos,Neg).
list_representation( Atom , rule(Atom,[],[]) ):- atom(Atom),!.


split_body( (not(F),Rest), Pos, [F|Neg] ):-
	!,split_body(Rest,Pos,Neg).
split_body( (F,Rest), [F|Pos], Neg ):-
	!,split_body(Rest,Pos,Neg).
split_body( (not(F)), [], [F] ):-!.
split_body( true, [], [] ):-!. 
split_body( (F), [F], [] ):-!.


%
%   choose(?Element, ?Set, ?Residue)
%   is true when Set is a list, Element occurs in Set, and Residue is
%   everything in Set except Element (things stay in the same order).
%

choose(X, [X|R],     R        ).
choose(X, [A,X|R],   [A|R]    ).
choose(X, [A,B,X|R], [A,B|R]  ).
choose(X, [A,B,C|L], [A,B,C|R]) :-
        choose(X, L, R).        


%
% combi(+[L1,..,Ln],T) true iff T is an n-tuple in (L1 X...X Ln)
% where backtracking it gives all n-tuples
%
combi( [], [] ) :- !.
combi( [L1|L], [E|T] ) :- 
	choose(E,L1,_),
	combi(L,T).

%
% basic io 
% write_blocksets(L)
%
write_blocksets( [] ) :- !.
write_blocksets( [blocksets(C,B)|R] ) :-
	write('Clause: '),
	write(C),write('.  Blocking sets :'),nl,
	write_list(B),nl,
	write_blocksets(R).
write_list( [] ) :- !.
write_list( [E|L] ) :- 
	write('     '),write(E),nl,
	write_list(L).

%
% lp2file(+LP,+File) true iff file File is output of LP  
% 
lp2file(LP,File):-
	ext2int(Ext,LP),
	write2file(File,Ext).

write2file(File,LP):-
	open(File,write,Stream),
	write_clauses(Stream,LP),
	close(Stream).

write_clauses(_,[]).
write_clauses(Stream,[H|T]):-
	write(Stream,H),
	writeln(Stream,"."),
	write_clauses(Stream,T).




