;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Skill level interface
;;; The following functions must be defined form the RAP execution layer to work.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *pending-events* nil)
(defvar *waiting-data* nil)
(defvar *primitive-time* 1)
(defvar *resources* nil)

;;; Primitive RAPs "enable" skills, meaning that they activate a motor function.
;;; Enable-primitive-skill gets called and skill-defn is bound to '(skill-name arg0 arg1 ...).
;;; I am not sure what enabled? is for, so I ignore it.
;;; Enable-primitive-skill should call an appropriate handler function.  The handler function is
;;; responsible for generating some result symbol.  I define 'okay as meaning the motor skill
;;; was performed successfully.  Any other result symbol must therefore be an error message.
;;; Enable-primitive-skill takes care of setting the success or failure event on the *waiting-data*
;;; queue.
;;; Under most circumstances, enable-primitive-skill should return t.
#+OLD
(defun enable-primitive-skill (skill-defn enabled?)
   (let* ((result (apply (first skill-defn) (rest skill-defn)))
          (new-data (if (eq result 'okay)
                              (list 'succeeded (first skill-defn) result)
                              (list 'failed (first skill-defn) result))))
      (setf *waiting-data* (append *waiting-data* (list new-data)))
      t))
 
(defun enable-primitive-skill (skill-defn enabled?)
   (multiple-value-bind (result success) (apply (first skill-defn) (rest skill-defn))
      (setf *waiting-data* (append *waiting-data* (list (if success
                                                           (list 'succeeded (first skill-defn) result)
                                                           (list 'failed (first skill-defn) result)))))
      
      t))

;;; Skills can be continuous motor functions, so must be disabled in robots and other real systems.
;;; For a simulated system, disable-primitive-skill does nothing.
(defun disable-primitive-skill (skill-name)
   (values))

;;; Fetch-skill-resources returns a list of resources that a motor skill uses.  The resources is
;;; a list of symbols which means nothing to the RAP execution system, but is used to compare skills
;;; to make sure two skills that require the same resource are not activated at the same time.
(defun fetch-skill-resources (skill-name)
   (rest (assoc skill-name *resources*)))


;;; Enable-primitive-event is called when the system is expecting an event to occur in the 
;;; skill level.  For example, a primitive RAP might declare a (wait-for) clause.  The skill-level
;;; records these expected events in the *pending-events* list.
(defun enable-primitive-event (event-id repeat event-defn event-values)
   (let ((key (if (atom event-defn) event-defn (first event-defn)))
         (rest (if (atom event-defn) nil (rest event-defn))))
      (push (list key event-id repeat rest) *pending-events*)
      (values)))


;;; Disable-primitive-event removes an event from the *pending-events* list when that event is no
;;; longer expected to occur.  The event-id is a RAP execution level identifier.
(defun disable-primitive-event (event-id)
   (setf *pending-events* (delete-if #'(lambda (x) (eq (second x) event-id)) *pending-events*))
   (values))

;;; Fetch-primitive-events polls the skill-level for events that may have occured.  For example
;;; the arm-move skill may have succeeded; this is an event of consequence.  Actual events are 
;;; detected in the *waiting-data* list and matched up with events that the RAP execution system
;;; is expecting.  RAP expects events when they are declared in the (wait-for) clause in a primitive.
;;; When an event that is expected is found, the waiting-data event is renamed to its RAP execution
;;; level name.  The *waiting-data* queue is returned to the RAP execution level, hopefully fully
;;; transformed into terms RAP will understand.  
;;; If the event is :single, it will only occur once, so it is not expected to occur again, so it
;;; is removed from the *pending-events* list via disable-primitive-event.
;;; The queue is erased after being polled.
(defun fetch-primitive-events ()
   (dolist (datum *waiting-data*)
      (let ((event (find-if #'(lambda (x) (and (eq (car datum) (car x))
                                               (every #'eq (cadddr x) (cdr datum))))
                     *pending-events*)))
         (when event
            (setf (car datum) (cadr event))
            (setf (cdr datum) (nthcdr (length (cadddr event)) (cdr datum)))
            (if (eq (caddr event) :single)
               (disable-primitive-event (cadr event))))))
   (let ((data *waiting-data*))
      (setf *waiting-data* nil)
      data))

;;; Query-primitive-value allows the RAP exeuction level to query values of variables on the
;;; skill-level.  I don't know how to do this.
(defun query-primitive-value (query-defn query-values)
   (cond ((listp query-defn) (apply (first query-defn) (rest query-defn)))
         (t (apply query-defn nil))))
 
;;; Fetch-primitive-time returns the system time from the skill-level.  This could be a true counter
;;; (measured in seconds from 1900) or simulator time in ticks.
(defun fetch-primitive-time ()
   (get-internal-run-time))

;;; Let-primitive-time-pass is a way for the RAP system to wait for time to pass.  In a real system
;;; where time is measured by the passage of time, this function should wait until the specified
;;; amount of time has passed before returning.  For a simulator system where time is measured in
;;; ticks, it is safe to increment the simulator time the given number of ticks.
(defun let-primitive-time-pass (amount)
   (sleep (* 0.01 amount)))

