/**
 * @class nReadValues
 * file name: nReadValues.h
 * @author Betti Oesterholz
 * @date 27.02.2006
 * @mail webmaster@bernd-oesterholz.name
 *
 * System: C++
 *
 * This class represents a basic univers.
 * Copyright (C) @c GPL3 2008  Betti Oesterholz
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @description:
 * This class is for reading values, like numbers or path, from an 
 * parameter file.
 */
/*
History:
11.01.2009  Oesterholz  refactoring for doxygen
*/


#ifndef ___READVALUES_h__
#define ___READVALUES_h__

#include <fstream>
#include <cstdio>


using std::ifstream;
using std::fstream;

/** this is a namespace for reading values from an parameterfile */
namespace nReadValues{

/** @def PATH_SEPERATOR the operatingsystem dependent seperator in filepath */
#ifdef linux
	#define PATH_SEPERATOR      '/'
	#define NOT_PATH_SEPERATOR  '\\'
#else//dos/windows
	#define PATH_SEPERATOR      '\\'
	#define NOT_PATH_SEPERATOR  '/'
#endif

	/**
	 * makes the char c to a integer number c represents
	 * @param  cDigit the character to convert
	 * @return the integer number the char c represents (e.g. "1"=1) or -1 if it
	 * 	isn't a number
	 */
	int charToInt( const char cDigit );

	/** reads a character substring from the given stream ifstrInput that
	 * is terminated with ':' or ';'
	 * @param ifstrInput the input stream to read the char string from
	 * @return the readed string terminated with 0;
	 * 	or NULL if a error occured
	 */
	char* readChar( ifstream &ifstrInput );

	/**
	 * reads a path from the given stream ifstrInput;
	 * the string for the path should be terminated with ':' or ';'
	 * (this character will be deleted);
	 * if pathE is true the path will be terminated with '/' (the method 
	 * insert '/' at the end of the path if it needs to)
	 * @param ifstrInput the input stream to read the path from
	 * @param pathE if the path should be terminated with '/'
	 * @return the readed path string terminated with 0;
	 * 	or NULL if a error occured
	 */
	char* readPath( ifstream &ifstrInput, bool pathE=false );

	/**
	 * reads a double number from the given inputstream;
	 * the number string should be terminated with ':' or ';',
	 * it will be read till (inclusive) ther terminating character
	 * @param ifstrInput the input stream to read the double number from
	 * @return the readed double number
	 */
	double readDouble( ifstream &ifstrInput );

	/**
	 * reads a integer number from the given inputstream;
	 * the number string should be terminated with ':' or ';',
	 * it will be read till (inclusive) ther terminating character
	 * @param ifstrInput the input stream to read the integer number from
	 * @return the readed integer number
	 */
	long readInt( ifstream &ifstrInput );

	/**
	 * reads a truth value from the given inputstream;
	 * strings for true are: t, true, y, yes, w, wahr, j, ja, 1
	 * all other strings are readed as false;
	 * all strings can be writen in upper cases, just the first char of the words
	 * are checked;
	 * the truth value string should be terminated with ':' or ';',
	 * it will be read till (inclusive) ther terminating character
	 * @param ifstrInput the input stream to read the truth value from
	 * @return  the readed truth value
	 */
	bool readBool( ifstream &ifstrInput );

	/**
	 * reads a time period from the given inputstream;
	 * the period string should be terminated with ':' or ';',
	 * it will be read till (inclusive) ther terminating character
	 * @param ifstrInput the input stream to read the time period from
	 * @return the readed time period as a time_t
	 */
	time_t readPeriod( ifstream &ifstrInput );

	/**
	 * checks if a file for the given path exists and can be opened
	 * @param path a string for the path to be cheked
	 * @return true if the path exists, else false
	 */
	bool checkPath( char* path );
	
	/**
	 * if the path folders are not existing, this function creats them
	 * @param path the string of the path that should be existing
	 */
	void cratePath( char* path );

	/**
	 * searches in a tabel for the line with the given number or the next 
	 * grater number in front and lets the fstream point to the end of the 
	 * line;
	 * the rows in the tabel must end with a ";";
	 * if no data is in the file, the pointers of the stream are set to the 
	 * file begin;
	 * if ther is data in the file, but not front number greater or equal 
	 * the given number, the file pointers are set to the end of the file
	 * @param ifstrInput the stream str to search in
	 * @param lNumber the number to search for 
	 * @param uiRows the number of rows in the tabel in the stream
	 * @return true if the number was found and the stream output pointer 
	 * 	pointing to the end of the line wich begins with number, or false
	 */
	bool searchForLine(fstream *ifstrInput, long lNumber, unsigned int uiRows);
	

};

#endif




